/** @file   waypointcontroller.cpp
 * @brief   Implementation of WaypointController - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#include "waypointcontroller.h" // class's header file
#include "www_assert.h"
#include "utils.h"

using eng2d::Vec2D;

namespace WeWantWar {


///
/// Constructor, destructor
/// =======================

/** Constructor
 */
WaypointController::WaypointController( GameObject* pObj ) :
  BaseController( pObj ),
  m_waypointmode( MODE_LOOP ),
  m_activeWaypoint( 0 ),
  m_tolerance( 30 ),
  m_dir( 1 ),
  m_route( "unnamed" )
{
}

/** Destructor
 */
WaypointController::~WaypointController()
{
	
}



///
/// Methods
/// =======


/** Sets the route this controller should follow.
 */
void WaypointController::setRoute( const Route& rRoute )
{
  m_route = rRoute;
  m_activeWaypoint = 0;
}


/** Sets the mode
 */
void WaypointController::setMode( Mode m )
{
  m_waypointmode = m;
}



/** Sets the tolerance
 */
void WaypointController::setTolerance( float tolerance )
{
  WWW_ASSERT( tolerance > 1 );
  m_tolerance = tolerance;
}



/** Sets the direction.
 */
void WaypointController::direction( int dir )
{
  WWW_ASSERT( abs(dir) == 1 );
  
  m_dir = dir;
}



/** Sets the next active waypoint
 */
void WaypointController::setNextWaypoint( int nextWP )
{
  m_activeWaypoint = nextWP;
}




/** Goes to next waypoint.
 */
void WaypointController::step()
{
  m_activeWaypoint += this->direction();
  if ( m_activeWaypoint >= m_route.nodeCount() ) {
    // The final node have been reached. Stop here, loop or pingpong
    switch ( this->getMode() ) {
      case ( MODE_ONCE ): {
        // Stop here.
        m_activeWaypoint -= this->direction();
        return;
      }
      case ( MODE_LOOP ): {
        // Go back to first node
        m_activeWaypoint = 0;
        return;
      }
      case ( MODE_PINGPONG ): {
        // Do pingpong
        m_activeWaypoint = m_route.nodeCount() - 2;
        this->direction( -1 );
        return;
      }
    }
  } else if ( m_activeWaypoint < 0 ) {
    // The first node reached while playing the route backwards. This can
    // happen only when we're doing a pingpong play
    m_activeWaypoint = 0;
    this->direction( 1 );
  }
}



///
/// Getter methods
/// ==============

/** Returns the current waypoint
 */
int WaypointController::getActiveWaypoint() const
{
  return m_activeWaypoint;
}



/** Returns the mode
 */
WaypointController::Mode WaypointController::getMode() const
{
  return m_waypointmode;
}



/** Returns the tolerance
 */
float WaypointController::getTolerance() const
{
  return m_tolerance;
}



/** Returns the direction.
 */
int WaypointController::direction() const
{
  return m_dir;
}



/** Returns the route this controller has.
 */
const Route& WaypointController::getRoute() const
{
  return m_route;
}



/** Tells if we're having a valid route.
 */
bool WaypointController::validRoute() const
{
  return ( m_route.nodeCount() > 1 ) ? true : false;
}



/** Tells if we've reached our current waypoint.
 */
bool WaypointController::waypointReached() const
{
  float f = m_route.distanceFromWaypoint( this->getGameObject()->position(),
                                          this->getActiveWaypoint() );
                                    
  if ( f < this->getTolerance() ) {
    return true;
  }
  return false;
}



/** Tells if our GameObject is towards the current waypoint.
 */
bool WaypointController::isFacingWaypoint( float factor ) const
{
  Vec2D wpPos( m_route.getWaypointPos( m_activeWaypoint ) );
  return Utils::isFacingCoordinate( this->getGameObject(), wpPos, factor );
}



/** Finds the shortest turning direction.
 */
int WaypointController::findShortestTurn() const
{
  Vec2D wpPos( m_route.getWaypointPos( m_activeWaypoint ) );
  GameObject* pG = this->getGameObject();
  return Utils::findTurningDir( pG->position(), wpPos, pG->angle() );
}


} // end of namespace

/**
 * Version history
 * ===============
 * $Log: waypointcontroller.cpp,v $
 * Revision 1.1.1.1  2006/01/21 23:02:44  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:15:38+02  lamminsa
 * Initial revision
 *
 */
 
